    /*
     * slides: array of all paint + quiz data injected from PHP
     * idx: current slide index
     */

    // Get the slides data from PHP
    const slides = window.slides || [];

    // Initialize current slide index
    let idx = 0;

    /**
     * Render a specific slide in the quiz panel
     * @param {number} i index of the slide
     */
    function render(i) {
        idx = i;
        const s = slides[i];

        // Check if we're in quiz mode or product view
        const quizPanel = document.getElementById('quiz');
        const storeGrid = document.querySelector('.store-grid');

        if (quizPanel && quizPanel.style.display !== 'none') {
            // Quiz mode
            // Update image & details
            document.getElementById('paint-img').src = s.paintPhoto;
            document.getElementById('paint-name').textContent = s.paintName;
            document.getElementById('paint-price').textContent = s.price;
            document.getElementById('paint-grams').textContent = s.grams;
            document.getElementById('paint-shine').textContent = s.shine;
            document.getElementById('paint-base').textContent = s.base;

            // Hide next button & tips on new render
            document.getElementById('next').style.display = 'none';
            document.getElementById('tips-container').style.display = 'none';

            // Set mission text
            document.getElementById('mission-text').innerHTML =
                `Mission: Find the correct RAL color <strong>${s.mainColorName}</strong>` +
                ` for <em>${s.paintName}</em>.`;

            // Build poll of swatches
            const sw = document.getElementById('ral-swatches');
            sw.innerHTML = '';
            s.opts.forEach(o => {
                const d = document.createElement('div');
                d.className = 'swatch';
                d.style.backgroundColor = o.hex;
                d.textContent = o.num;
                d.onclick = () => answer(d, o.num === s.correct);
                sw.appendChild(d);
            });
        } else if (storeGrid) {
            // Product view mode - do nothing as the products are already displayed
            console.log('Product view mode - no need to render');
        }
    }

    /**
     * Handle swatch selection
     * @param {HTMLElement} el the clicked swatch
     * @param {boolean} ok whether it's correct
     */
    function answer(el, ok) {
        if (ok) {
            el.classList.add('correct'); // reveal color
            document.getElementById('next').style.display = 'block';
            alert('Good job!');
        } else {
            alert('Wrong, try again.');
        }
    }

    // Wire up the Next → button to go to next slide
    document.getElementById('next').onclick = () => render((idx + 1) % slides.length);

    // Give Tips button shows hint list
    document.getElementById('give-tips').onclick = () => {
        const tips = slides[idx].tips;
        const list = document.getElementById('tips-list');
        list.innerHTML = '';
        if (tips.length) {
            tips.forEach(x => {
                const li = document.createElement('li');
                li.textContent = x;
                list.appendChild(li);
            });
            document.getElementById('tips-container').style.display = 'block';
        } else {
            document.getElementById('tips-container').style.display = 'none';
            alert('No tips available.');
        }
    };

    // Initial render of first paint
    if (slides.length > 0) {
        render(0);
    }

    // Add filter form functionality
    document.addEventListener('DOMContentLoaded', function() {
        console.log('DOM Content Loaded');
        
        const filterForm = document.getElementById('filter-form');
        console.log('Filter form element:', filterForm);
        
        if (!filterForm) {
            console.error('Filter form not found!');
            return;
        }

        console.log('Filter form found, adding submit listener');
        
        filterForm.addEventListener('submit', async function(e) {
            // Prevent the default form submission
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Form submitted');
            
            try {
                // Get all form data and convert it to URL parameters
                const formData = new FormData(this);
                const params = new URLSearchParams();
                
                // Log each form field value
                for (let [key, value] of formData.entries()) {
                    console.log(`Form field ${key}:`, value);
                    if (value) {  // Only add non-empty values
                        params.append(key, value);
                    }
                }
                
                const queryString = params.toString();
                console.log('Filter parameters:', queryString);
                
                // Get the store grid element and show loading message
                const storeGrid = document.querySelector('.store-grid');
                if (!storeGrid) {
                    console.error('Store grid not found!');
                    return;
                }
                
                // Show loading state
                storeGrid.innerHTML = '<div class="loading">Loading...</div>';
                
                // Send request to filter_products.php with our parameters
                const url = 'filter_products.php' + (queryString ? '?' + queryString : '');
                console.log('Sending request to:', url);
                
                const response = await fetch(url, {
                    method: 'GET',
                    headers: {
                        'Accept': 'application/json'
                    }
                });
                
                console.log('Response status:', response.status);
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const text = await response.text();
                console.log('Raw response:', text);
                
                let products;
                try {
                    products = JSON.parse(text);
                } catch (e) {
                    console.error('Failed to parse JSON:', e);
                    console.error('Raw response was:', text);
                    throw new Error('Invalid JSON response from server');
                }
                
                console.log('Parsed products:', products);
                
                // Clear the loading message
                storeGrid.innerHTML = '';
                
                // If no products match the filters, show a message
                if (!products || products.length === 0) {
                    storeGrid.innerHTML = '<div class="no-results">No products found matching your criteria</div>';
                    return;
                }
                
                // For each product in the results, create and add its HTML
                products.forEach((item, index) => {
                    const productHtml = `
                        <a class="link-detail" href="detail.php?">
                            <div class="store-item" data-index="${index}">
                                
                                    <img  src="${item.photo || 'testimagepaint.png'}" alt="${item.name || 'Product Image'}" onerror="this.src='testimagepaint.png'">
                               
                                <h3>${item.name || 'Unnamed Product'}</h3>
                                <p>€${item.price || '0'}</p>
                                <p>${item.grams ? item.grams + 'g' : ''}</p>
                                <div class="swatch-list">
                                    ${item.rals ? item.rals.map(ral => 
                                        `<div class="swatch" style="background:${ral.hex || '#CCC'}; width:16px; height:16px;"></div>`
                                    ).join('') : ''}
                                </div>
                            </div>
                        </a>
                    `;
                    storeGrid.innerHTML += productHtml;
                });
            } catch (error) {
                console.error('Error in filter submission:', error);
                const storeGrid = document.querySelector('.store-grid');
                if (storeGrid) {
                    storeGrid.innerHTML = `<div class="error">Error: ${error.message}</div>`;
                }
            }
            
            // Prevent form submission
            return false;
        });

        // Add reset button functionality
        const resetButton = filterForm.querySelector('button[type="reset"]');
        if (resetButton) {
            resetButton.addEventListener('click', function(e) {
                e.preventDefault();
                filterForm.reset();
                // Trigger the filter form submission to show all products
                filterForm.dispatchEvent(new Event('submit'));
            });
        }
    });